package com.pdftools.pdf2imgapi;

import com.pdftools.NativeLibrary;
import com.pdftools.Stream;

/**
 * The NativeLibrary class provides the basic functions for interfacing to a native (JNI) library.
 */
public class Pdf2Img extends NativeLibrary
{
    /**
     * Set the license key.
     * @param licenseKey The license key
     * @return true if the license is valid.
     */
    public static native boolean setLicenseKey(String licenseKey);

    /**
     * Check if a valid license is loaded.
     * @return true if a valid license is loaded.
     */
    public static native boolean getLicenseIsValid();

     /**
     * Get the version of the product.
     * @return a string in the format 'A.B.C.D'.
     */
    public static native String getProductVersion(); 

    /**
     * Constructor
     */
    public Pdf2Img()
    {
        createObject();
    }

    /**
     * @deprecated Deprecated in Version 4.7, replaced by {@link #open(String, String)}
     */
    public Pdf2Img(String strFileName, String strPassword)
    {
        createObject();
        open(strFileName, strPassword);
    }

    /**
     * @deprecated Deprecated in Version 4.7, replaced by {@link #openMem(byte[], String)}
     */
    public Pdf2Img(byte[] pData, String strPassword)
    {
        createObject();
        openMem(pData, strPassword);
    }

	/**
	 * This method sets the path to the sRGB profile. If no path is set, the
	 * profile called bin/Icc/sRGB.icm is used if available. If no color profile
	 * can be found, the color version is applied algorithmically.
	 * 
	 * @param sFileName
	 *            File name of the RGB profile.
	 * @return true if the color profile could successfully be selected, false
	 *         otherwise.
	 */
    public static native boolean setsRGBProfile(String sFileName);

	/**
	 * This method sets the path to the CMYK profile. If no path is set, the
	 * profile called bin/Icc/CMYK.icc is used if available. If no color profile
	 * can be found, the color version is applied algorithmically.
	 * 
	 * @param sFileName
	 *            File name of the CMYK profile
	 * @return true if the color profile could successfully be selected, false
	 *         otherwise.
	 */
    public static native boolean setCMYKProfile(String sFileName);

    /**
	 * Set the name of the Color Management System (CMS) engine (None,
	 * Neugebauer, lcms or a filename). Choose among the available color
	 * management engines to transform colors.
	 * <p>
	 * "None": If this switch is set to none then the algorithms specified in
	 * the PDF reference are used. This results in the maximum possible
	 * contrast.
	 * <p>
	 * "Neugebauer": The Neugebauer algorithm efficiently converts CMYK to RGB.
	 * It does not need any color profiles. The results, however, look similar
	 * to conversion using color profiles.
	 * <p>
	 * "lcms" (default): Use ICC color profiles. Default profiles are used for
	 * all unmanaged device color spaces.
	 * <p>
	 * FileName: Providing a file name, a configurable version of the Neugebauer
	 * algorithm is applied. The coefficients can be defined in the text file.
	 * The Neugebauer algorithm mixes the colors based on the amount of color
	 * and the corresponding weighted coefficient. Altering the values for a
	 * pure color specifically changes the result for this pure color. The color
	 * transition remains smooth.
	 * 
	 * @param sCMSEngine
	 *            Name of the CMS Engine
	 * @return true on success
     */
    public static native boolean setCMSEngine(String sCMSEngine);

	/**
    * Set handle for the process.
    */
    private native boolean createObject();

	/**
    * Release handle for the process.
    */
	public native void destroyObject();

	/**
	 * Open a PDF random access disk file. It makes the objects contained in the
	 * PDF document accessible. If the document is already open, it is closed
	 * first.
	 * 
	 * @param strFileName
	 *            the file name and optionally the file path.
	 * @param strPassword
	 *            the user or the owner password of the encrypted PDF document.
	 *            If this parameter is left out an empty string is used as a
	 *            default.
	 * @return True: The file could be successfully opened. False: The file does
	 *         not exist, it is corrupt, or the password is not valid. Use the
	 *         property {@link #getErrorCode getErrorCode} for additional
	 *         information.
	 */
    public native boolean open(String strFileName, String strPassword);

	/**
	 * Open a PDF random access disk file. It makes the objects contained in the
	 * PDF document accessible. If the document is already open, it is closed
	 * first.
	 * 
     * @param inputStream
     *            the PDF file as a stream
	 * @param strPassword
	 *            the user or the owner password of the encrypted PDF document.
	 *            If this parameter is left out an empty string is used as a
	 *            default.
	 * @return True: The file could be successfully opened. False: The file does
	 *         not exist, it is corrupt, or the password is not valid. Use the
	 *         property {@link #getErrorCode getErrorCode} for additional
	 *         information.
	 */
    public native boolean openStream(Stream inputStream, String strPassword);

	/**
	 * Closes the currently open input file(s). If the document is already
	 * closed the method does nothing.
	 */
    public native boolean close();

	/**
    * Create an image file to render pages to.
    * @param strFileName the image file name and optionally the file path.
    * The file name defines the image format. Supported extensions are: 
    * .bmp, .gif, .jb2, .jpg, .jpeg, .jp2, .jpf, .jpx, .png, .tif, .tiff
    * @return True: The file was created successfully. False otherwise.
    */
    public native boolean  createImage(String strFileName);

	/**
	 * Create an image in memory as a byte array.
	 * 
	 * @param strExtension
	 *            the extension of the image format. For a list of supported
	 *            extensions see method {@link #createImage createImage}. The
	 *            leading "." needs to be included.
	 * @see #getImage
	 * @return True: The image could successfully be created. False: The image
	 *         could not be created.
    */
    public native boolean createImageInMemory(String strExtension);

	/**
	 * Get the image in memory as a byte array which was previously saved using
	 * {@link #createImageInMemory createImageInMemory}.
	 */
    public native byte[] getImage();

	/**
	 * Close an open image document. If the document is already closed the
	 * method does nothing.
	 */
    public native boolean closeImage();
    
	/**
	 * Check if a page contains color.
	 * @param iPageNo the page number to check
	 * @return true if the page contains color, false otherwise.
	 */
    public native boolean hasColor(int iPageNo);

	/**
    * Render the selected page number in the PDF document to the created raster
    * image.
    * @param iPageNumber The page number in the PDF file to be rendered.
    * @return True: Page rendered successfully. False: The page could not be
    *         rendered (possible reasons: out of range, no PDF opened, no image
    *         file created).
    */
    public native boolean renderPage(int iPageNumber);

	/**
    * Set a page for reading out page properties
    * Default: last rendered page
    * @param iPageNumber The page number in the PDF file to be set.
    * @return True: Page could be set successfully. False: The page could not
    *         be set (possible reasons: out of range, no PDF opened).
    */
    public native boolean  setPage(int iPageNumber);

    /**
    * Set the resolution in the X-axis of the image in DPI's (dots per inch).
    * Default: 150
    * @param newVal the DPI value to be set.
    */
    public native void  setXDPI(float newVal);

	/**
    * Get the resolution in the X-axis of the image in DPI's (dots per inch).
    * @return the resolution in the X-axis of the image in DPI's.
    */
    public native float getXDPI();

	/**
    * Set the resolution in the Y-axis of the image in DPI's (dots per inch).
    * Default: 150
    * @param newVal the DPI value to be set.
    */
    public native void  setYDPI(float newVal);

	/**
    * Get the resolution in the Y-axis of the image in DPI's (dots per inch).
    * @return the resolution in the Y-axis of the image in DPI's.
    */
    public native float getYDPI();

	/**
	 * Set the the bits per pixel (bi-tonal: 1, gray scale: 8, RGB true color:
	 * 24, CMYK: 32).
	 * <p>
	 * When using 1 bit per pixel, it is suggested to disable anti-aliasing
	 * (disable {@link #setOptions eOptionHighQuality}) and set a suitable
	 * dithering algorithm ({@link #setDithering setDithering}).
	 * <p>
	 * Default: 24
	 */
    public native void setBitsPerPixel(int newVal);

	/**
	 * Get the the bits per pixel (bi-tonal: 1, gray scale: 8, RGB true color:
	 * 24, CMYK: 32).
	 * <p>
	 * When using 1 bit per pixel, it is suggested to disable anti-aliasing
	 * (disable {@link #setOptions eOptionHighQuality}) and set a suitable
	 * dithering algorithm ({@link #setDithering setDithering}).
	 */
    public native int getBitsPerPixel();
    
    /**
	 * Set the threshold for converting from gray to bi-tonal when
	 * {@link #getDithering getDithering} is eDitherNone. The value must be in the
	 * range of 0 to 255, default is 181.
     */
    public native void setBilevelThreshold(int newVal);

	/**
	 * Get the the threshold for converting from gray to bi-tonal when
	 * {@link #getDithering getDithering} is eDitherNone.
	 */
    public native int getBilevelThreshold();

	/**
	 * Set the dithering mode. Dithering refers to the procedure of simulating
	 * colors or grayscales. This is mainly useful for low color depth (e.g.
	 * black and white or indexed) images. Default: eFloydSteinberg.
	 * 
	 * @see com.pdftools.NativeLibrary.DITHERINGMODE
	 */
    public native void  setDithering(int newVal);

	/**
	 * Get the dithering mode. Dithering refers to the procedure of simulating
	 * colors or grayscales. This is mainly useful for low color depth (e.g.
	 * black and white or indexed) images. Default: eFloydSteinberg.
	 * 
	 * @see com.pdftools.NativeLibrary.DITHERINGMODE
	 */
    public native int getDithering();

	/**
	 * Get the total number of pages of an open document. If the document is
	 * closed or if the document is a collection (also known as PDF Portfolio)
	 * then this property is {@code null}.
	 */
    public native int getPageCount();

	/**
	 * Set the rotate mode: Define if and how to rotate the page before
	 * rendering.
	 * 
	 * Default: eRotateAttribute.
	 * 
	 * @see com.pdftools.NativeLibrary.ROTATEMODE
	 */
    public native void setRotateMode(int newVal);

	/**
	 * Get the rotate mode: Get whether and how the page is rotated before
	 * rendering.
	 * 
	 * Default: eRotateAttribute.
	 * 
	 * @see com.pdftools.NativeLibrary.ROTATEMODE
	 */
    public native int getRotateMode();

	/**
	 * Set the compression type of TIFF images. For any other image format, the
	 * compression is automatically defined by the file extension (the file
	 * name). Default: eComprLZW
	 * 
	 * @see com.pdftools.NativeLibrary.COMPRESSION
	 */
    public native void setCompression(int newVal);

	/**
	 * Get the compression type of TIFF images. For any other image format, the
	 * compression is automatically defined by the file extension (the file
	 * name). Default: eComprLZW
	 * 
	 * @see com.pdftools.NativeLibrary.COMPRESSION
	 */
    public native int getCompression();

	/**
	 * @deprecated Deprecated in Version 4.7, replaced by {@link #setCenter(boolean)}
	 */
    public void setCenter(int bNewVal)
    {
        setCenter(bNewVal != 0);
    }

	/**
	 * Set the center mode. When set to True, the document is horizontally and
	 * vertically centered on the page. When set to False, the document is
	 * printed to the upper left corner of the page. Default: False.
	 * 
	 * @param center
	 *            True: The page is horizontally and vertically centered. 
     *            False: The page is rendered to the upper left corner.
	 */
    public native void setCenter(boolean center);

	/**
    * Get the center mode. Default: False.
    * @return the center mode: True: The page is horizontally and vertically centered.
    * False: The page is rendered to the upper left corner.
    */
    public native boolean getCenter();

	/**
	 * Set the offset of the image in points. The default offset is the
	 * CropBox's. Set the offset before {@link #renderPage renderPage} and get it
	 * afterwards.
	 * 
	 * @param x
	 *            The X offset of the image in points.
	 * @param y
	 *            The Y offset of the image in points.
	 */
    public native void setPageOffs(float x, float y);

	/**
	 * Get the X offset of the image in points. The default offset is the
	 * CropBox's. Set the offset before {@link #renderPage renderPage} and get
	 * it afterwards.
	 */
    public native float getPageXOffs();

	/**
	 * Get the Y offset of the image in points. The default offset is the
	 * CropBox's. Set the offset before {@link #renderPage renderPage} and get
	 * it afterwards.
	 */
    public native float getPageYOffs();

	/**
    * Set the dimensions of the image in points.
    * @param x The X dimension of the image in points.
    * @param y The Y dimension of the image in points.
    */
    public native void setPageSize(float x, float y);
	
	/**
     * Get the width of the page in points.
     * @return the width of the page in points.
     */
    public native float getPageWidth();

    /**
     * Get the height of the page in points.
     * @return the height of the page in points.
     */
    public native float getPageHeight();
    
	/**
    * Set the fit page mode. Default: True.
    * @param bNewVal true: the page is scaled to fit the image (in either width or height), 
    * false: the page is rendered with its true size.
    */
    public native void setFitPage(int bNewVal);

	/**
    * Get the fit page mode. Default: True.
    * @return true: the page is scaled to fit the image (in either width or height), 
    * false: the page is rendered with its true size.
    */
    public native boolean getFitPage();

	/**
    * Get the error code for a previously failed function call.
    * @see com.pdftools.NativeLibrary.ERRORCODE
    */
    public native int getErrorCode();

    /**
	 * Get the error message text associated with the last error (see property
	 * {@link #getErrorCode getErrorCode}).
	 * 
	 * @return The message describing the last error that occurred while
	 *         processing.
	 */
    public native String getErrorMessage();

	/**
	 * Set the quality index of the lossy compression. This is a value from 1 to
	 * 100. This can be applied for JPEG and JPEG2000 compression. For JPEG2000,
	 * a quality index of 100 means lossless compression. JPEG compression is
	 * always lossy. Default:80
	 */
    public native void setImageQuality(int newVal);

	/**
	 * Get the quality index of the lossy compression. This is a value from 1 to
	 * 100. This can be applied for JPEG and JPEG2000 compression. For JPEG2000,
	 * a quality index of 100 means lossless compression. JPEG compression is
	 * always lossy. Default:80
	 */
    public native int getImageQuality();

    /**
     * Set the dimensions of the image in pixel.
     * @param cx The X dimension of the image in points.
     * @param cy The Y dimension of the image in points.
     */
    public native void setBitmapDimensions(int cx, int cy);
	
    /**
     * Get the bitmap width in pixels.
     * @return the width in pixels.
     */
    public native int getBitmapWidth();

    /**
     * Get the bitmap height in pixels.
     * @return the height in pixels.
     */
    public native int getBitmapHeight();
    
	/**
	 * Set the preserve aspect ratio mode. If True a uniform up- or down-scaling
	 * is applied, i.e. the output image has the same ratio of width to height
	 * as the input file and its size will fit into the defined dimensions,
	 * given by {@link #setBitmapDimensions setBitmapDimensions}. Default:
	 * False.
	 */
    public native void setPreserveAspectratio(boolean bNewVal);

	/**
	 * Get the preserve aspect ratio mode. If True a uniform up- or down-scaling
	 * is applied, i.e. the output image has the same ratio of width to height
	 * as the input file and its size will fit into the defined dimensions,
	 * given by {@link #setBitmapDimensions setBitmapDimensions}. Default:
	 * False.
	 */
    public native boolean getPreserveAspectratio();

	/**
	 * Set the resolution of the image in DPI (dots per inch). Both the
	 * resolutions for the x- and y-axis are set to the same value. Setting
	 * setDPI is redundant to setting the specialized properties {@link #setXDPI
	 * setXDPI} and {@link #setYDPI setYDPI}.
	 */
    public native void setDPI(float bNewVal);

	/**
	 * Get the resolution of the image in DPI (dots per inch). Returns the
	 * square root of the product of the resolutions in x and y direction.
	 */
    public native float getDPI();

	/**
	 * Open a PDF stream from an in-memory file. It makes the objects contained
	 * in the PDF document accessible. If the document is already open it is
	 * closed first.
	 * 
	 * @param pData
	 *            the memory block containing the PDF file given as a one
	 *            dimensional byte array
	 * @param strPassword
	 *            the password to be used to decrypt a password protected input
	 *            PDF. If this parameter is left out an empty string is used as
	 *            a default.
	 * @return True: the document could be successfully opened. False: The
	 *         document could not be opened, it is corrupt, or the password is
	 *         not valid.
	 */
    public native boolean openMem(byte[] pData, String strPassword);

	/**
	 * Convert a complete PDF file to a (multi-page) TIFF image file.
	 * 
	 * @param strInputFileName
	 *            the PDF file name and optionally the file path.
	 * @param strOutputFileName
	 *            the image file name and optionally the file path.
	 * @param strPassword
	 *            the user or the owner password of the encrypted PDF document.
	 *            If this parameter is left out an empty string is used as a
	 *            default.
	 * @return True: The file was converted successfully. False: The PDF file
	 *         does not exists, it is corrupt, the password is invalid, or the
	 *         image file is locked.
	 */
    public native boolean convertFile(String strInputFileName, String strOutputFileName, String strPassword);

	/**
	 * Set the standard setting for Fax machines (TIFF Type F settings).
	 * <p>
	 * Equal to RotateMode = RotatePortrait, SetBitmapDimensions(1728, 0), XDPI
	 * = 204, YDPI = 98, Compression = eComprGroup3.
	 * 
	 * @return True: The fax mode could be set.
	 */
    public native boolean faxSSetting();

	/**
	 * Set the high resolution setting for Fax machines (TIFF Type F settings).
	 * <p>
	 * Equal to RotateMode = RotatePortrait, SetBitmapDimensions(1728, 0), XDPI
	 * = 204, YDPI = 196, Compression = eComprGroup3.
	 * 
	 * @return True: The fax mode could be set.
	 */
    public native boolean faxHSetting();


	/**
	 * Set the bit fill order. 1 is MSB (Most Significant Bit) first, 2 is LSB
	 * (Least significant bit) first. Default: 1
	 */
    public native void setFillOrder(int iFillOrder);


	/**
	 * Get the bit fill order. 1 is MSB (Most Significant Bit) first, 2 is LSB
	 * (Least significant bit) first. Default: 1
	 */
    public native int getFillOrder();

	/**
	 * Set the post filter ratio 1:n.
	 * <p>
	 * This property is used to enable and parameterize super-sampling, a
	 * technique to initially render the image at a higher resolution and then
	 * sample it down to the target resolution. As a result of that process the
	 * final image appears smoother, i.e. anti-aliased.
	 * <p>
	 * Applying super-sampling improves the image quality when rendering at low
	 * target resolutions (72 dpi or less); the higher the target resolution the
	 * less the visual impact.
	 * <p>
	 * This property requires memory and CPU time quadratically to the ratio,
	 * therefore only small values, such as 2 or 3 should be used. If a too high
	 * value (in combination with the original image size) is set, it is
	 * ignored.
	 * <p>
	 * Default: 1.
	 */
    public native void setFilterRatio(int nFilterRatio);


	/**
	 * Get the post filter ratio 1:n.
	 * <p>
	 * This property is used to enable and parameterize super-sampling, a
	 * technique to initially render the image at a higher resolution and then
	 * sample it down to the target resolution. As a result of that process the
	 * final image appears smoother, i.e. anti-aliased.
	 * <p>
	 * Applying super-sampling improves the image quality when rendering at low
	 * target resolutions (72 dpi or less); the higher the target resolution the
	 * less the visual impact.
	 * <p>
	 * This property requires memory and CPU time quadratically to the ratio,
	 * therefore only small values, such as 2 or 3 should be used. If a too high
	 * value (in combination with the original image size) is set, it is
	 * ignored.
	 * <p>
	 * Default: 1.
	 */
    public native int getFilterRatio();

	/**
	 * Set the color space of the image. For black-white bi-tonal images, a gray
	 * color space must be selected.
	 * <p>
	 * Default: eColorRGB
	 * 
	 * @see com.pdftools.NativeLibrary.COLORSPACE
	 */
    public native void setColorSpace(int iColorSpace);


	/**
	 * Get the color space of the image. For black-white bi-tonal images, a gray
	 * color space must be selected.
	 * <p>
	 * Default: eColorRGB
	 * 
	 * @see com.pdftools.NativeLibrary.COLORSPACE
	 */
    public native int getColorSpace();


	/**
	 * @deprecated Deprecated in Version 4.7, replaced by {@link #setOptions2(int)}
	 */
    public native void setOptions(int iOptions);

	/**
	 * @deprecated Deprecated in Version 4.7, replaced by {@link #getOptions2()}
	 */
    public native int getOptions();

	/**
	 * Set the rendering options. Use bitwise "OR" to add an option. Use bitwise
	 * "AND NOT" to remove an option.
	 * 
	 * @see com.pdftools.NativeLibrary.RENDEREROPTION2
	 */
    public native void setOptions2(int iOptions);

	/**
	 * Get the rendering options.
	 * 
	 * @see com.pdftools.NativeLibrary.RENDEREROPTION2
	 */
    public native int getOptions2();

	/**
	 * Set a rendering property.
	 * 
	 * @see com.pdftools.NativeLibrary.RENDERINGPROPERTY
	 */
    public native boolean setRenderingProperty(int property, String value);

	/**
	 * Reset all rendering properties to their default value.
	 * 
	 */
    public native void clearRenderingProperties();

	/**
	 * Get the minimum hinting size.
	 * 
	 */
    public native int getMinHintingSize();

	/**
	 * Set the minimum hinting size.
	 * 
	 */
    public native void setMinHintingSize(int iOptions);

	/**
	 * Get the unsupported features used on the last page rendered.
	 */
    public native int getUnsupportedFeatures();

	/**
	 * Get the number of optional content groups (also known as "layers") of the
	 * document.
	 */
    public native int getOcgCount();

	/**
	 * Retrieve an optional content group object.
	 * 
	 * @param iOcg
	 *            Number of the optional content group. Optional content groups
	 *            are numbered from 0 to {@link #getOcgCount getOcgCount}-1.
	 */
    public native Ocg getOcg(int iOcg);

    protected static native void initialize(String strVersion);

    protected long _handle;
    private Object _ocg;

    /** load the native library */
    static 
    {
        try 
        {
            initialize(VERSION);
        } 
        catch (UnsatisfiedLinkError ue) 
        {
            System.loadLibrary("Pdf2ImgAPI");
            initialize(VERSION);
        }
    }

    /**
    * The standard finalize method frees internal native resources
    * in case the destroyObject() call is missing in the application.
    */
	protected void finalize() throws Throwable
	{
		destroyObject();
	}

}
